<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\Query\SelectQuery;
use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;

/**
 * Menus Model
 *
 * @property \App\Model\Table\PageContentsTable&\Cake\ORM\Association\HasMany $PageContents
 * @property \App\Model\Table\PageProductsTable&\Cake\ORM\Association\HasMany $PageProducts
 *
 * @method \App\Model\Entity\Menu newEmptyEntity()
 * @method \App\Model\Entity\Menu newEntity(array $data, array $options = [])
 * @method array<\App\Model\Entity\Menu> newEntities(array $data, array $options = [])
 * @method \App\Model\Entity\Menu get(mixed $primaryKey, array|string $finder = 'all', \Psr\SimpleCache\CacheInterface|string|null $cache = null, \Closure|string|null $cacheKey = null, mixed ...$args)
 * @method \App\Model\Entity\Menu findOrCreate($search, ?callable $callback = null, array $options = [])
 * @method \App\Model\Entity\Menu patchEntity(\Cake\Datasource\EntityInterface $entity, array $data, array $options = [])
 * @method array<\App\Model\Entity\Menu> patchEntities(iterable $entities, array $data, array $options = [])
 * @method \App\Model\Entity\Menu|false save(\Cake\Datasource\EntityInterface $entity, array $options = [])
 * @method \App\Model\Entity\Menu saveOrFail(\Cake\Datasource\EntityInterface $entity, array $options = [])
 * @method iterable<\App\Model\Entity\Menu>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\Menu>|false saveMany(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\Menu>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\Menu> saveManyOrFail(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\Menu>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\Menu>|false deleteMany(iterable $entities, array $options = [])
 * @method iterable<\App\Model\Entity\Menu>|\Cake\Datasource\ResultSetInterface<\App\Model\Entity\Menu> deleteManyOrFail(iterable $entities, array $options = [])
 */
class MenusTable extends Table
{
    /**
     * Initialize method
     *
     * @param array<string, mixed> $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('menus');
        $this->setDisplayField('name');
        $this->setPrimaryKey('id');

        $this->belongsTo('ParentMenus', [
            'foreignKey' => 'parent_menu_id',
            'className' => 'Menus'
        ]);
        $this->hasMany('ChildMenus', [
            'foreignKey' => 'parent_menu_id',
            'className' => 'Menus',
            'conditions' => [
                'ChildMenus.status' => 1
            ]
        ]);
        $this->hasMany('SubChildMenus', [
            'foreignKey' => 'parent_menu_id',
            'className' => 'Menus',
            'conditions' => [
                'SubChildMenus.status' => 1
            ]
        ]);
        $this->hasMany('Albums', [
            'foreignKey' => 'menu_id',
        ]);
        $this->hasMany('Contents', [
            'foreignKey' => 'menu_id',
        ]);
        $this->hasMany('PageContents', [
            'foreignKey' => 'menu_id',
        ]);
        $this->hasMany('PageProducts', [
            'foreignKey' => 'menu_id',
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator): Validator
    {
        $validator
            ->scalar('name')
            ->maxLength('name', 255)
            ->requirePresence('name', 'create')
            ->notEmptyString('name');

        $validator
            ->scalar('description')
            ->maxLength('description', 255)
            ->allowEmptyString('description');

        $validator
            ->integer('parent_menu_id')
            ->notEmptyString('parent_menu_id');

        $validator
            ->integer('status')
            ->notEmptyString('status');

        $validator
            ->scalar('slug')
            ->maxLength('slug', 255)
            ->requirePresence('slug', 'create')
            ->notEmptyString('slug');

        $validator
            ->integer('show_on_home')
            ->allowEmptyString('show_on_home');

        $validator
            ->integer('show_on_footer')
            ->notEmptyString('show_on_footer');

        $validator
            ->scalar('image_path')
            ->maxLength('image_path', 255)
            ->allowEmptyString('image_path');

        $validator
            ->scalar('content')
            ->maxLength('content', 255)
            ->allowEmptyString('content');

        $validator
            ->integer('home_order')
            ->requirePresence('home_order', 'create')
            ->notEmptyString('home_order');

        return $validator;
    }

    function getParentMenus()
    {
        return $this->find()
            ->where(['Menus.parent_menu_id' => 0, 'Menus.status' => 1])
            ->contain(['ChildMenus'])
            ->order(['home_order' => 'ASC'])
            ->toArray();
    }

    function getFooterMenus()
    {
        return $this->find()
            ->where(['Menus.parent_menu_id' => 0, 'Menus.status' => 1, 'Menus.show_on_footer' => 1,])
            ->contain(['ChildMenus'])
            ->order(['home_order' => 'ASC'])
            ->toArray();
    }

    function getAboutUsContent()
    {
        return $this->find()
            ->select(['content'])
            ->where(['slug' => 'about-us'])
            ->first();
    }
}
