<?php
/**
 * Routes configuration.
 *
 * In this file, you set up routes to your controllers and their actions.
 * Routes are very important mechanism that allows you to freely connect
 * different URLs to chosen controllers and their actions (functions).
 *
 * It's loaded within the context of `Application::routes()` method which
 * receives a `RouteBuilder` instance `$routes` as method argument.
 *
 * CakePHP(tm) : Rapid Development Framework (https://cakephp.org)
 * Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @copyright     Copyright (c) Cake Software Foundation, Inc. (https://cakefoundation.org)
 * @link          https://cakephp.org CakePHP(tm) Project
 * @license       https://opensource.org/licenses/mit-license.php MIT License
 */

use Cake\Routing\Route\DashedRoute;
use Cake\Routing\RouteBuilder;

/*
 * This file is loaded in the context of the `Application` class.
 * So you can use  `$this` to reference the application class instance
 * if required.
 */
return function (RouteBuilder $routes): void {
    /*
     * The default class to use for all routes
     *
     * The following route classes are supplied with CakePHP and are appropriate
     * to set as the default:
     *
     * - Route
     * - InflectedRoute
     * - DashedRoute
     *
     * If no call is made to `Router::defaultRouteClass()`, the class used is
     * `Route` (`Cake\Routing\Route\Route`)
     *
     * Note that `Route` does not do any inflections on URLs which will result in
     * inconsistently cased URLs when used with `{plugin}`, `{controller}` and
     * `{action}` markers.
     */
    $routes->setRouteClass(DashedRoute::class);

    $routes->scope('/', function (RouteBuilder $builder): void {
        /*
         * Here, we are connecting '/' (base path) to a controller called 'Pages',
         * its action called 'display', and we pass a param to select the view file
         * to use (in this case, templates/Pages/home.php)...
         */
        $builder->connect('/', ['controller' => 'Home', 'action' => 'index']);

        /*
         * ...and connect the rest of 'Pages' controller's URLs.
         */

        /**
         * Routing for About Us
         * 
         * 1. Main menu
         * 2. Sub menu
         * 3. Detailed view of a selected record
         */
        $builder->connect(
            '/about-us',
            [
                'controller' => 'AboutUs',
            ]
        );

        $builder->connect(
            '/about-us/{subsection}',
            [
                'controller' => 'AboutUs',
                'action' => 'subMenu',
            ],
            [
                'pass' => ['subsection']
            ]
        );

        $builder->connect(
            '/about-us/details/{id}',
            [
                'controller' => 'AboutUs',
                'action' => 'details',
            ],
            [
                'pass' => ['id']
            ]
        );


        /**
         * Routing for the gallery
         * Route for gallery home and album 
         */
        $builder->connect(
            '/gallery',
            [
                'controller' => 'Gallery',
            ]
        );

        $builder->connect(
            '/gallery/details/{id}',
            [
                'controller' => 'Gallery',
                'action' => 'details',
            ],
            [
                'pass' => ['id']
            ]
        );

        /**
         * Routing for the contact us
         * Route for contact us
         */
        $builder->connect(
            '/contact-us',
            [
                'controller' => 'ContactUs',
            ]
        );

        $builder->connect(
            '/contact-us/contact-form',
            [
                'controller' => 'ContactUs',
                'action' => 'contactForm'
            ]
        );

        /**
         * Set the fallback controller to contents, when no matching controller is found
         * from the matching the route
         * Basic contents supports
         * 1. Main menu
         * 2. Sub menu
         * 3. Detailed view of a selected record
         */
        $builder->connect(
            '/{section}',
            [
                'controller' => 'Content',
            ]
        );

        $builder->connect(
            '/{section}/{subsection}',
            [
                'controller' => 'Content',
                'action' => 'subsection',
            ]
        );

        $builder->connect(
            '/{section}/details/{id}',
            [
                'controller' => 'Content',
                'action' => 'details',
            ],
            [
                'pass' => ['id']
            ]
        );

        /*
         * Connect catchall routes for all controllers.
         *
         * The `fallbacks` method is a shortcut for
         *
         * ```
         * $builder->connect('/{controller}', ['action' => 'index']);
         * $builder->connect('/{controller}/{action}/*', []);
         * ```
         *
         * You can remove these routes once you've connected the
         * routes you want in your application.
         */
        $builder->fallbacks();
    });

    /*
     * If you need a different set of middleware or none at all,
     * open new scope and define routes there.
     *
     * ```
     * $routes->scope('/api', function (RouteBuilder $builder): void {
     *     // No $builder->applyMiddleware() here.
     *
     *     // Parse specified extensions from URLs
     *     // $builder->setExtensions(['json', 'xml']);
     *
     *     // Connect API actions here.
     * });
     * ```
     */
};
